// COLLECTOR.JS
// -------------------------------------------------------------------------------

// NodeJS profiling ----
// require('look').start();
//----------------------

var fs = require('fs');
var os = require('os');
var exec = require('child_process').exec;
var ip_pub = require('public-ip');
var stdio = require('stdio');

// CMDLINE ARGS -------------------------------------------------------------------
var cmdopts = stdio.getopt({
  'mode': {key: 'm', args: 1, default: 'console', description: '[console|fork|mysql]'}}
);

if(cmdopts.mode === 'console'){
  function dataSend(data){console.log(data);}
} else if(cmdopts.mode === 'fork'){
  // (!) This function must be declared as variable
  var dataSend = function(data){process.send(data);}
} else {
  console.log('Unknown/unsupported mode option, process exit...');
  process.exit();
}

// STARTS COLLECTOR MODULES --------------------------------------------------------
//random(500);
info(0);
uptime(60000)
cpu(200);
memory(1000);
loadavg(1000);
network(500);
storage(10000);

// COLECTOR MODULES ----------------------------------------------------------------
function info(delay) {

  var hostname = os.hostname();
  if (hostname.length > 10) {hostname = hostname.substring(0,8) + '~'};

  var kernel = os.type() + ' ' + os.release() + ' ' + os.arch();
  var os_release = fs.readFileSync('/etc/os-release', 'utf8');
  var distrib = os_release.match('PRETTY_NAME="(.*)"')[1];
  var cpu = os.cpus()[0].model;

  var memory = os.totalmem() / 1000000000;
  if (memory < 1) { memory = (parseInt(memory*10))/10 + ' GB' }
  else { memory = parseInt(memory) + ' GB' };

  var ip_lan = os.networkInterfaces();
  for (i in ip_lan) {
    if(!ip_lan[i][0].internal) {
      ip_lan = ip_lan[i][0].address;
      break;
    }
  }

  ip_pub.v4().then(ip_wan => {
    dataSend([
      'info',
      { 'hostname': hostname,
      'kernel': kernel,
      'distrib': distrib,
      'cpu': cpu,
      'memory': memory,
      'ip_lan': ip_lan,
      'ip_wan': ip_wan
    }
  ]);
});
}

function uptime(delay) {
  var uptime, hrs, min;

  function procedure(delay){
    uptime = new Date(os.uptime()*1000);
    hrs = (os.uptime()/3600).toString().split('.')[0];
    min = uptime.getMinutes();

    setTimeout(function(){ procedure(delay); }, delay);
    dataSend(['uptime', [hrs, min]]);
  }
  procedure(delay);
}

function memory(delay) {

  var fileData, MemTotal, MemFree, MemAvailable, SwapTotal, SwapFree,
  MemUsage, MemFullUsage, SwapUsage;

  function procedure(delay){

    fileData = fs.readFileSync('/proc/meminfo', 'utf8');

    MemTotal = fileData.match('MemTotal' + ':\\s*(\\d*)')[1];
    MemFree = fileData.match('MemFree' + ':\\s*(\\d*)')[1];
    MemAvailable = fileData.match('MemAvailable' + ':\\s*(\\d*)')[1];
    SwapTotal = fileData.match('SwapTotal' + ':\\s*(\\d*)')[1];
    SwapFree = fileData.match('SwapFree' + ':\\s*(\\d*)')[1];

    MemUsage = parseInt(100 - ((100 / MemTotal) * MemAvailable));
    MemFullUsage = parseInt(100 - ((100 / MemTotal) * MemFree) - MemUsage);
    SwapUsage = parseInt(100 - ((100 / SwapTotal) * SwapFree));

    setTimeout(function(){ procedure(delay); }, delay);
    dataSend(['memory', {'MemUsage': MemUsage, 'MemFullUsage': MemFullUsage, 'SwapUsage': SwapUsage}]);
  }
  procedure(delay);
}

function cpu(delay) {

  var statFile, infoFile, CPUStat, CpuTotal, CpuUsage, CpuFreq,
  user=[],nice=[],system=[],idle=[],iowait=[],irq=[],softirq=[];

  var CpuCore = parseInt(fs.readFileSync('/proc/cpuinfo', 'utf8').match('siblings\\s*:\\s(\\d*)')[1]);

  for (var core = 0; core < (CpuCore+1); core++) {
    user.push([0,0]); nice.push([0,0]); system.push([0,0]); idle.push([0,0]);
    iowait.push([0,0]); irq.push([0,0]); softirq.push([0,0]);
  }

  function procedure(delay){

    CpuUsage = [], CpuFreq = [];
    infoFile = fs.readFileSync('/proc/cpuinfo', 'utf8');
    statFile = fs.readFileSync('/proc/stat', 'utf8');

    CpuFreq = infoFile.match(/cpu MHz\s*:\s*\d*/g);
    CpuFreq[CpuCore] = 0;

    for (core = 0; core < (CpuCore+1); core++) {

      if (core == (CpuCore)) {
        var cpuName = 'cpu';
        CpuFreq[core] = parseInt(CpuFreq[core] / CpuCore);
      } else {
        var cpuName = 'cpu' + core;
        CpuFreq[core] = CpuFreq[core].match('\\d*$')[0];
        CpuFreq[CpuCore] += parseInt(CpuFreq[core]);
      }

      CPUStat = statFile.match(cpuName+'\\s*(\\d*)\\s*(\\d*)\\s*(\\d*)'
      + '\\s*(\\d*)\\s*(\\d*)\\s*(\\d*)\\s*(\\d*)');

      user[core].shift(); user[core].push(parseInt(CPUStat[1]));
      nice[core].shift(); nice[core].push(parseInt(CPUStat[2]));
      system[core].shift(); system[core].push(parseInt(CPUStat[3]));
      idle[core].shift(); idle[core].push(parseInt(CPUStat[4]));
      iowait[core].shift(); iowait[core].push(parseInt(CPUStat[5]));
      irq[core].shift(); irq[core].push(parseInt(CPUStat[6]));
      softirq[core].shift(); softirq[core].push(parseInt(CPUStat[7]));

      CpuTotal = (user[core][1] - user[core][0])
      + (nice[core][1] - nice[core][0])
      + (system[core][1] - system[core][0])
      + (idle[core][1] - idle[core][0])
      + (iowait[core][1] - iowait[core][0])
      + (irq[core][1] - irq[core][0])
      + (softirq[core][1] - softirq[core][0]);

      CpuUsage.push(parseInt(100-(((idle[core][1]-idle[core][0])*100)/CpuTotal)));
    }

    setTimeout(function(){ procedure(delay); }, delay);
    dataSend(['cpu', {'CpuUsage': CpuUsage, 'CpuFreq': CpuFreq}]);
  }
  procedure(delay);
}

function loadavg(delay) {

  function procedure(delay){
    fileData = fs.readFileSync('/proc/loadavg', 'utf8');
    Values = fileData.match('(\\d*\\.\\d*)\\s*(\\d*\\.\\d*)\\s*(\\d*\\.\\d*)');

    setTimeout(function(){ procedure(delay); }, delay);
    dataSend(['loadavg', [Values[1], Values[2], Values[3]]]);
  }
  procedure(delay);
}

function network(delay) {

  var inBytes = [0,0], outBytes = [0,0], netUp, netDown;
  var firstIgnore = 1;

  // Get default network interface
  exec("ip route list", function(error, stdout, stderr) {
    var iface = stdout.match(/default.*dev\s([\w\d]*)/)[1];

    function procedure(delay){

      // TCP/UDP connections count
      conns = {'tcp':'', 'udp':''};
      for(proto in conns){
        connData = fs.readFileSync('/proc/net/' + proto, 'utf8');
        lines = connData.match(/.*\n/g).length-1;
        conns[proto] = lines;
      }
      // ---------------------------------------

      // Network load
      netData = fs.readFileSync('/proc/net/dev', 'utf8');
      Values = netData.match('\\s*' + iface + ':'
      + '\\s*(\\d*)' + '(\\s*\\d*){7}' + '\\s*(\\d*).*');

      inBytes.shift(); inBytes.push(parseInt(Values[1]));
      outBytes.shift(); outBytes.push(parseInt(Values[3]));

      if (firstIgnore) {
        firstIgnore = 0; netDown = 0; netUp = 0;
      } else {
        netDown = (inBytes[1] - inBytes[0]) * (1000 / delay);
        netUp = (outBytes[1] - outBytes[0]) * (1000 / delay);
      }
      // ----------------------------------------

      setTimeout(function(){ procedure(delay); }, delay);
      dataSend(['network', {
        'iface': iface,
        "down": netDown,
        'up': netUp,
        'tcp': conns.tcp,
        'udp': conns.udp}
      ]);
    }
    procedure(delay);
  });
}

function storage(delay) {

  var disk, values, result = [];

  function procedure(delay){

    exec("df -h", function(error, stdout, stderr) {

      result = [];
      var output = stdout.match(/.*\n/g);

      for (var line in output) {
        var dfparse = /\/dev\/(\S+)\s+(\S+)\s+\S+\s+\S+\s+(\S+)%\s+(\S+)\n/
        values = output[line].match(dfparse);

        if(values){
          volume = {};
          volume.device = values[1];
          volume.capacity = values[2];
          volume.usage = values[3];
          volume.mount = values[4];
          result.push(volume);
        }
      }

      setTimeout(function(){ procedure(delay); }, delay);
      dataSend(['storage', result]);
    });
  }
  procedure(delay);
}
